// SPDX-License-Identifier: MIT
pragma solidity ^0.8.19;

import "@openzeppelin/contracts/access/AccessControl.sol";
import "@openzeppelin/contracts/security/ReentrancyGuard.sol";

contract AdvancedVault is AccessControl, ReentrancyGuard {
    bytes32 public constant PROPOSER_ROLE = keccak256("PROPOSER_ROLE");
    bytes32 public constant CANCELLER_ROLE = keccak256("CANCELLER_ROLE");
    
    uint256 public constant APPROVAL_THRESHOLD = 2;
    uint256 public constant TRANSACTION_EXPIRY_DURATION = 7 days;

    struct Transaction {
        address to;
        uint256 value;
        bytes data;
        bool executed;
        uint256 expiry;
        mapping(address => bool) approvals;
        uint256 approvalCount;
    }

    Transaction[] public transactions;
    
    event Deposit(address indexed sender, uint256 amount);
    event Proposal(uint256 indexed txId);
    event Approval(uint256 indexed txId, address indexed owner);
    event Execution(uint256 indexed txId);
    event Cancellation(uint256 indexed txId);

    constructor(address[] memory proposers, address[] memory cancellers) {
        _grantRole(DEFAULT_ADMIN_ROLE, msg.sender);
        for (uint256 i = 0; i < proposers.length; i++) {
            _grantRole(PROPOSER_ROLE, proposers[i]);
        }
        for (uint256 i = 0; i < cancellers.length; i++) {
            _grantRole(CANCELLER_ROLE, cancellers[i]);
        }
    }

    receive() external payable {
        emit Deposit(msg.sender, msg.value);
    }

    function proposeTransaction(address to, uint256 value, bytes calldata data)
        external
        onlyRole(PROPOSER_ROLE)
    {
        uint256 txId = transactions.length;
        transactions.push(Transaction({
            to: to,
            value: value,
            data: data,
            executed: false,
            expiry: block.timestamp + TRANSACTION_EXPIRY_DURATION,
            approvalCount: 0
        }));
        emit Proposal(txId);
    }

    function approveTransaction(uint256 txId)
        external
        onlyRole(PROPOSER_ROLE)
        nonReentrant
    {
        Transaction storage transaction = transactions[txId];
        require(txId < transactions.length, "Transaction not found");
        require(!transaction.executed, "Transaction already executed");
        require(block.timestamp < transaction.expiry, "Transaction expired");
        require(!transaction.approvals[msg.sender], "Already approved");

        transaction.approvals[msg.sender] = true;
        transaction.approvalCount++;
        emit Approval(txId, msg.sender);
    }

    function executeTransaction(uint256 txId)
        external
        onlyRole(PROPOSER_ROLE)
        nonReentrant
    {
        Transaction storage transaction = transactions[txId];
        require(transaction.approvalCount >= APPROVAL_THRESHOLD, "Not enough approvals");
        require(!transaction.executed, "Transaction already executed");
        
        transaction.executed = true;
        (bool success, ) = transaction.to.call{value: transaction.value}(transaction.data);
        require(success, "Execution failed");
        
        emit Execution(txId);
    }

    function cancelTransaction(uint256 txId)
        external
        onlyRole(CANCELLER_ROLE)
    {
        Transaction storage transaction = transactions[txId];
        require(!transaction.executed, "Transaction already executed");
        
        transaction.executed = true; // Mark as executed to prevent future actions
        emit Cancellation(txId);
    }

    function getTransaction(uint256 txId)
        external
        view
        returns (address, uint256, bytes memory, bool, uint256, uint256)
    {
        Transaction storage tx_ = transactions[txId];
        return (tx_.to, tx_.value, tx_.data, tx_.executed, tx_.expiry, tx_.approvalCount);
    }
}