from flask import Flask, jsonify, render_template
from web3 import Web3
import requests

app = Flask(__name__)

# Using Compound's public API as an example
COMPOUND_API_URL = "https://api.compound.finance/api/v2/governance/proposals"

class GovernanceAnalyzer:
    def __init__(self):
        self.proposals = self.fetch_proposals()

    def fetch_proposals(self):
        try:
            response = requests.get(COMPOUND_API_URL)
            response.raise_for_status()
            return response.json().get("proposals", [])
        except requests.RequestException:
            return []
    
    def analyze_proposals(self, limit=5):
        analysis_results = []
        for prop in self.proposals[:limit]:
            total_votes = float(prop.get("for_votes", 0)) + float(prop.get("against_votes", 0))
            if total_votes > 0:
                for_percentage = (float(prop.get("for_votes", 0)) / total_votes) * 100
            else:
                for_percentage = 0
            
            analysis_results.append({
                "id": prop.get("id"),
                "title": prop.get("title"),
                "state": prop.get("state"),
                "for_votes": f"{float(prop.get('for_votes', 0)):,.0f}",
                "against_votes": f"{float(prop.get('against_votes', 0)):,.0f}",
                "for_percentage": f"{for_percentage:.2f}%"
            })
        return analysis_results

analyzer = GovernanceAnalyzer()

@app.route('/')
def index():
    return render_template('index.html')

@app.route('/api/proposals')
def get_proposals():
    return jsonify(analyzer.analyze_proposals())

if __name__ == '__main__':
    app.run(debug=True, port=5001)